"""
optimal bandwidth selection (Scott's Rule) in multivariate varying coefficient model (MVCM).

Author: Chao Huang (chaohuang.stat@gmail.com)
Last update: 2017-09-18
"""

import numpy as np
from numpy.linalg import inv
from statsmodels.robust.scale import mad

"""
installed all the libraries above
"""


def bw_rt(coord_mat, x_design, y_design):
    """
        optimal bandwidth (Scott's Rule) selection in MVCM.

        :param
            coord_mat (matrix): common coordinate matrix (l*d)
            x_design (matrix): non-genetic design matrix (n*p)
            y_design (matrix): imaging response data (response matrix, n*l*m)
        :return
            h_rt (matrix): optimal bandwidth matrix (l*d)
            hat_mat (matrix): hat matrix (n*n)
    """

    # Set up
    n, p = x_design.shape
    l, d = coord_mat.shape
    m = y_design.shape[2]
    h_rt = np.zeros((d, m))

    # calculate the hat matrix
    hat_mat = np.dot(np.dot(x_design, inv(np.dot(x_design.T, x_design)+np.eye(p)*0.000001)), x_design.T)

    # calculate the median absolute deviation (mad) on both coordinate data and response data

    for mii in range(m):
        z = np.reshape(y_design[:, :, mii], (y_design.shape[0]*y_design.shape[1], 1))
        h_y = mad(z)    # type: np.ndarray
        # h_y_median = np.median(h_y)
        for dii in range(d):
            h_coord = mad(coord_mat[:, dii])
            # h_rt[dii, mii] = 1.06*(1/l)**(1/(d+4))*np.sqrt(h_coord*h_y)
            h_rt[dii, mii] = 1.06*(1/n)**(1/(d+4))*h_coord
    return h_rt, hat_mat
